/**
 * Phone Number Formatting Utilities
 */

/**
 * Format phone number for WhatsApp
 * @param {string} phone - Phone number
 * @returns {string} Formatted phone number with @s.whatsapp.net
 */
export const formatPhone = (phone) => {
  if (!phone) return null;

  // Already formatted
  if (phone.endsWith('@s.whatsapp.net')) {
    return phone;
  }

  // Remove all non-digit characters
  let formatted = phone.replace(/\D/g, '');

  // Add WhatsApp suffix
  return `${formatted}@s.whatsapp.net`;
};

/**
 * Format group ID for WhatsApp
 * @param {string} groupId - Group ID
 * @returns {string} Formatted group ID with @g.us
 */
export const formatGroup = (groupId) => {
  if (!groupId) return null;

  // Already formatted
  if (groupId.endsWith('@g.us')) {
    return groupId;
  }

  // Keep digits and hyphens only
  let formatted = groupId.replace(/[^\d-]/g, '');

  // Add group suffix
  return `${formatted}@g.us`;
};

/**
 * Extract phone number from JID
 * @param {string} jid - WhatsApp JID
 * @returns {string} Phone number without suffix
 */
export const extractPhone = (jid) => {
  if (!jid) return null;

  // Remove @s.whatsapp.net or @c.us
  return jid.replace(/@s\.whatsapp\.net|@c\.us/g, '').split(':')[0];
};

/**
 * Validate phone number format
 * @param {string} phone - Phone number
 * @returns {boolean} True if valid
 */
export const isValidPhone = (phone) => {
  if (!phone) return false;

  const cleaned = phone.replace(/\D/g, '');

  // Should be between 10-15 digits
  return cleaned.length >= 10 && cleaned.length <= 15;
};

export default {
  formatPhone,
  formatGroup,
  extractPhone,
  isValidPhone,
};
