import SessionManager from '../core/SessionManager.js';
import logger from '../utils/logger.js';
import { formatPhone } from '../utils/phoneFormatter.js';
import { delay } from '../utils/retry.js';
import { downloadMediaMessage } from '@whiskeysockets/baileys';
import axios from 'axios';
import fs from 'fs';
import path from 'path';
import { nanoid } from 'nanoid';
import baileysConfig from '../config/baileys.config.js';

/**
 * Message Service - Handles all message operations
 *
 * Enterprise Anti-Ban Protection:
 * - Random delays between messages (configurable min/max)
 * - Extended pauses after batch counts
 * - Typing indicators simulation
 * - Read receipts simulation
 * - Message queue management
 * - Daily/hourly limits tracking
 */
class MessageService {
  constructor() {
    // Track message counts per session for anti-ban protection
    this.sessionCounters = new Map();

    // Anti-ban configuration defaults (can be overridden per request)
    this.antiBanDefaults = {
      minDelay: 1,           // Minimum delay between messages (seconds)
      maxDelay: 3,           // Maximum delay between messages (seconds)
      delayAfterCount: 50,   // Apply extended delay after this many messages
      delayAfterDuration: 30, // Extended delay duration (seconds)
      resetAfterCount: 200,  // Reset counter after this many messages
      simulateTyping: true,  // Simulate typing indicator
      typingDuration: 2000,  // Typing indicator duration (ms)
      maxMessagesPerHour: 60, // Maximum messages per hour (recommended)
      maxMessagesPerDay: 200, // Maximum messages per day (recommended)
    };
  }

  /**
   * Get or initialize session counter for anti-ban tracking
   */
  getSessionCounter(sessionId) {
    if (!this.sessionCounters.has(sessionId)) {
      this.sessionCounters.set(sessionId, {
        count: 0,
        hourlyCount: 0,
        dailyCount: 0,
        lastReset: Date.now(),
        lastHourReset: Date.now(),
        lastDayReset: Date.now(),
      });
    }

    const counter = this.sessionCounters.get(sessionId);
    const now = Date.now();

    // Reset hourly counter
    if (now - counter.lastHourReset > 3600000) {
      counter.hourlyCount = 0;
      counter.lastHourReset = now;
    }

    // Reset daily counter
    if (now - counter.lastDayReset > 86400000) {
      counter.dailyCount = 0;
      counter.lastDayReset = now;
    }

    return counter;
  }

  /**
   * Calculate anti-ban delay based on configuration
   * Returns delay in milliseconds
   */
  calculateAntiBanDelay(sessionId, config = {}) {
    const settings = { ...this.antiBanDefaults, ...config };
    const counter = this.getSessionCounter(sessionId);

    counter.count++;
    counter.hourlyCount++;
    counter.dailyCount++;

    // Calculate base random delay
    const minMs = settings.minDelay * 1000;
    const maxMs = settings.maxDelay * 1000;
    let delayMs = Math.floor(Math.random() * (maxMs - minMs + 1)) + minMs;

    // Apply extended delay after batch count
    if (settings.delayAfterCount > 0 && counter.count % settings.delayAfterCount === 0) {
      const extendedMs = settings.delayAfterDuration * 1000;
      delayMs += extendedMs;

      logger.info(`Anti-ban: Extended delay applied`, {
        sessionId,
        messageCount: counter.count,
        totalDelay: delayMs,
        reason: `Every ${settings.delayAfterCount} messages`,
      });
    }

    // Reset counter if threshold reached
    if (settings.resetAfterCount > 0 && counter.count >= settings.resetAfterCount) {
      counter.count = 0;
      counter.lastReset = Date.now();

      logger.info(`Anti-ban: Counter reset`, {
        sessionId,
        resetAfterCount: settings.resetAfterCount,
      });
    }

    // Warn if approaching limits
    if (counter.hourlyCount > settings.maxMessagesPerHour * 0.8) {
      logger.warn(`Anti-ban: Approaching hourly limit`, {
        sessionId,
        hourlyCount: counter.hourlyCount,
        limit: settings.maxMessagesPerHour,
      });
    }

    if (counter.dailyCount > settings.maxMessagesPerDay * 0.8) {
      logger.warn(`Anti-ban: Approaching daily limit`, {
        sessionId,
        dailyCount: counter.dailyCount,
        limit: settings.maxMessagesPerDay,
      });
    }

    return delayMs;
  }

  /**
   * Simulate typing indicator for more human-like behavior
   */
  async simulateTyping(client, jid, duration = 2000) {
    try {
      await client.socket.sendPresenceUpdate('composing', jid);
      await delay(duration);
      await client.socket.sendPresenceUpdate('paused', jid);
    } catch (error) {
      // Non-critical, just log and continue
      logger.debug(`Could not simulate typing: ${error.message}`);
    }
  }

  /**
   * Mark chat as read for more natural behavior
   */
  async markAsRead(client, jid) {
    try {
      await client.socket.readMessages([{ remoteJid: jid }]);
    } catch (error) {
      // Non-critical, just log
      logger.debug(`Could not mark as read: ${error.message}`);
    }
  }
  /**
   * Check if socket is actually connected and healthy
   */
  isSocketHealthy(client) {
    if (!client || !client.socket) {
      return false;
    }

    // Check if we have a user (authenticated) - this is the most reliable indicator
    if (!client.socket.user) {
      return false;
    }

    // Check WebSocket ready state if available
    // Note: After Baileys updates, the ws object structure may vary
    try {
      const ws = client.socket.ws;
      if (ws && typeof ws.readyState !== 'undefined') {
        // 0 = CONNECTING, 1 = OPEN, 2 = CLOSING, 3 = CLOSED
        if (ws.readyState === 3) { // CLOSED
          return false;
        }
      }
    } catch (e) {
      // If we can't check WebSocket state, rely on user presence
      logger.debug(`Could not check WebSocket state: ${e.message}`);
    }

    return true;
  }

  /**
   * Verify session is healthy before operations
   * Returns the client if healthy, throws error otherwise
   */
  async verifySession(sessionId) {
    const client = SessionManager.get(sessionId);

    if (!client || !client.socket) {
      throw new Error('Session not found or not connected');
    }

    // First check if user exists (authenticated)
    if (!client.socket.user) {
      logger.warn(`Session ${sessionId} has no user - not authenticated`);
      throw new Error('Session is not authenticated. Please reconnect via QR code.');
    }

    // Check socket health - but be lenient
    if (!this.isSocketHealthy(client)) {
      logger.warn(`Session ${sessionId} appears unhealthy, but attempting operation anyway`);
      // Don't throw - try the operation anyway, Baileys will handle reconnection
    }

    return client;
  }

  /**
   * Check if number exists on WhatsApp with timeout protection
   */
  async checkNumberExists(sessionId, number) {
    try {
      const client = await this.verifySession(sessionId);

      const formattedNumber = formatPhone(number);

      // Add timeout for the onWhatsApp call (15 seconds instead of default 60)
      const timeoutPromise = new Promise((_, reject) => {
        setTimeout(() => reject(new Error('Number check timed out')), 15000);
      });

      const checkPromise = client.socket.onWhatsApp(formattedNumber);
      const [result] = await Promise.race([checkPromise, timeoutPromise]);

      return {
        exists: result?.exists || false,
        jid: result?.jid || null,
      };
    } catch (error) {
      logger.error(`Failed to check number: ${error.message}`, { sessionId, number });
      throw error;
    }
  }

  /**
   * Send text message with enterprise anti-ban protection
   *
   * @param {string} sessionId - WhatsApp session ID
   * @param {string} receiver - Recipient phone number
   * @param {string} text - Message text
   * @param {number} delayMs - Optional delay (if 0, uses calculated anti-ban delay)
   * @param {object} antiBanConfig - Optional anti-ban configuration from Laravel gateway settings
   */
  async sendTextMessage(sessionId, receiver, text, delayMs = 0, antiBanConfig = null) {
    try {
      // Verify session is healthy first
      const client = await this.verifySession(sessionId);

      const formattedReceiver = formatPhone(receiver);

      // Check if number exists (with 15s timeout)
      const check = await this.checkNumberExists(sessionId, receiver);
      if (!check.exists) {
        throw new Error('The receiver number does not exist on WhatsApp');
      }

      // Calculate anti-ban delay if not provided
      let actualDelay = delayMs;
      if (antiBanConfig) {
        actualDelay = this.calculateAntiBanDelay(sessionId, antiBanConfig);
      } else if (delayMs === 0) {
        actualDelay = this.calculateAntiBanDelay(sessionId);
      }

      // Apply delay before sending
      if (actualDelay > 0) {
        logger.debug(`Anti-ban: Waiting ${actualDelay}ms before sending`, { sessionId, receiver: formattedReceiver });
        await delay(actualDelay);
      }

      // Simulate typing for more human-like behavior (if enabled)
      const config = antiBanConfig || this.antiBanDefaults;
      if (config.simulateTyping !== false) {
        await this.simulateTyping(client, formattedReceiver, config.typingDuration || 2000);
      }

      // Send message with timeout protection (30 seconds)
      const sendPromise = client.socket.sendMessage(formattedReceiver, {
        text: text,
      });

      const timeoutPromise = new Promise((_, reject) => {
        setTimeout(() => reject(new Error('Message send timed out')), 30000);
      });

      const result = await Promise.race([sendPromise, timeoutPromise]);

      // Get counter stats for logging
      const counter = this.getSessionCounter(sessionId);

      logger.info(`Text message sent successfully`, {
        sessionId,
        receiver: formattedReceiver,
        messageId: result.key.id,
        antiBan: {
          delayApplied: actualDelay,
          sessionMessageCount: counter.count,
          hourlyCount: counter.hourlyCount,
          dailyCount: counter.dailyCount,
        },
      });

      return {
        message: 'The message has been successfully sent',
        messageId: result.key.id,
        timestamp: result.messageTimestamp,
        antiBanStats: {
          delayApplied: actualDelay,
          messageCount: counter.count,
          hourlyCount: counter.hourlyCount,
          dailyCount: counter.dailyCount,
        },
      };
    } catch (error) {
      logger.error(`Failed to send text message: ${error.message}`, { sessionId, receiver });
      throw error;
    }
  }

  /**
   * Send image message
   */
  async sendImageMessage(sessionId, receiver, imageUrl, caption = '', delayMs = 0) {
    try {
      const client = SessionManager.get(sessionId);

      if (!client || !client.socket) {
        throw new Error('Session not found or not connected');
      }

      const formattedReceiver = formatPhone(receiver);

      // Check if number exists
      const check = await this.checkNumberExists(sessionId, receiver);
      if (!check.exists) {
        throw new Error('The receiver number does not exist on WhatsApp');
      }

      // Download image if URL provided
      let imageBuffer;
      if (imageUrl.startsWith('http')) {
        const response = await axios.get(imageUrl, { responseType: 'arraybuffer' });
        imageBuffer = Buffer.from(response.data);
      } else if (imageUrl.startsWith('data:image')) {
        // Base64 image
        const base64Data = imageUrl.split(',')[1];
        imageBuffer = Buffer.from(base64Data, 'base64');
      } else {
        throw new Error('Invalid image URL or base64 format');
      }

      // Apply delay
      if (delayMs > 0) {
        await delay(delayMs);
      }

      // Send image
      const result = await client.socket.sendMessage(formattedReceiver, {
        image: imageBuffer,
        caption: caption,
      });

      logger.info(`Image message sent successfully`, {
        sessionId,
        receiver: formattedReceiver,
        messageId: result.key.id,
      });

      return {
        message: 'Image message sent successfully',
        messageId: result.key.id,
        timestamp: result.messageTimestamp,
      };
    } catch (error) {
      logger.error(`Failed to send image message: ${error.message}`, { sessionId, receiver });
      throw error;
    }
  }

  /**
   * Send video message
   */
  async sendVideoMessage(sessionId, receiver, videoUrl, caption = '', delayMs = 0) {
    try {
      const client = SessionManager.get(sessionId);

      if (!client || !client.socket) {
        throw new Error('Session not found or not connected');
      }

      const formattedReceiver = formatPhone(receiver);

      // Check if number exists
      const check = await this.checkNumberExists(sessionId, receiver);
      if (!check.exists) {
        throw new Error('The receiver number does not exist on WhatsApp');
      }

      // Download video
      let videoBuffer;
      if (videoUrl.startsWith('http')) {
        const response = await axios.get(videoUrl, { responseType: 'arraybuffer' });
        videoBuffer = Buffer.from(response.data);
      } else {
        throw new Error('Invalid video URL');
      }

      // Apply delay
      if (delayMs > 0) {
        await delay(delayMs);
      }

      // Send video
      const result = await client.socket.sendMessage(formattedReceiver, {
        video: videoBuffer,
        caption: caption,
      });

      logger.info(`Video message sent successfully`, {
        sessionId,
        receiver: formattedReceiver,
        messageId: result.key.id,
      });

      return {
        message: 'Video message sent successfully',
        messageId: result.key.id,
        timestamp: result.messageTimestamp,
      };
    } catch (error) {
      logger.error(`Failed to send video message: ${error.message}`, { sessionId, receiver });
      throw error;
    }
  }

  /**
   * Send audio message
   */
  async sendAudioMessage(sessionId, receiver, audioUrl, delayMs = 0) {
    try {
      const client = SessionManager.get(sessionId);

      if (!client || !client.socket) {
        throw new Error('Session not found or not connected');
      }

      const formattedReceiver = formatPhone(receiver);

      // Check if number exists
      const check = await this.checkNumberExists(sessionId, receiver);
      if (!check.exists) {
        throw new Error('The receiver number does not exist on WhatsApp');
      }

      // Download audio
      let audioBuffer;
      if (audioUrl.startsWith('http')) {
        const response = await axios.get(audioUrl, { responseType: 'arraybuffer' });
        audioBuffer = Buffer.from(response.data);
      } else {
        throw new Error('Invalid audio URL');
      }

      // Apply delay
      if (delayMs > 0) {
        await delay(delayMs);
      }

      // Send audio
      const result = await client.socket.sendMessage(formattedReceiver, {
        audio: audioBuffer,
        mimetype: 'audio/mpeg',
        ptt: false, // Set to true for voice note
      });

      logger.info(`Audio message sent successfully`, {
        sessionId,
        receiver: formattedReceiver,
        messageId: result.key.id,
      });

      return {
        message: 'Audio message sent successfully',
        messageId: result.key.id,
        timestamp: result.messageTimestamp,
      };
    } catch (error) {
      logger.error(`Failed to send audio message: ${error.message}`, { sessionId, receiver });
      throw error;
    }
  }

  /**
   * Send document message
   */
  async sendDocumentMessage(sessionId, receiver, documentUrl, filename, mimetype, delayMs = 0) {
    try {
      const client = SessionManager.get(sessionId);

      if (!client || !client.socket) {
        throw new Error('Session not found or not connected');
      }

      const formattedReceiver = formatPhone(receiver);

      // Check if number exists
      const check = await this.checkNumberExists(sessionId, receiver);
      if (!check.exists) {
        throw new Error('The receiver number does not exist on WhatsApp');
      }

      // Download document
      let documentBuffer;
      if (documentUrl.startsWith('http')) {
        const response = await axios.get(documentUrl, { responseType: 'arraybuffer' });
        documentBuffer = Buffer.from(response.data);
      } else {
        throw new Error('Invalid document URL');
      }

      // Apply delay
      if (delayMs > 0) {
        await delay(delayMs);
      }

      // Send document
      const result = await client.socket.sendMessage(formattedReceiver, {
        document: documentBuffer,
        fileName: filename,
        mimetype: mimetype || 'application/pdf',
      });

      logger.info(`Document message sent successfully`, {
        sessionId,
        receiver: formattedReceiver,
        messageId: result.key.id,
      });

      return {
        message: 'Document message sent successfully',
        messageId: result.key.id,
        timestamp: result.messageTimestamp,
      };
    } catch (error) {
      logger.error(`Failed to send document message: ${error.message}`, { sessionId, receiver });
      throw error;
    }
  }

  /**
   * Send button message
   * Note: WhatsApp has deprecated interactive buttons for non-business accounts.
   * This method now sends buttons as a formatted text message with numbered options.
   * For true interactive buttons, use WhatsApp Business API (Cloud API).
   */
  async sendButtonMessage(sessionId, receiver, text, buttons, footer = '', delayMs = 0) {
    try {
      const client = SessionManager.get(sessionId);

      if (!client || !client.socket) {
        throw new Error('Session not found or not connected');
      }

      const formattedReceiver = formatPhone(receiver);

      // Check if number exists
      const check = await this.checkNumberExists(sessionId, receiver);
      if (!check.exists) {
        throw new Error('The receiver number does not exist on WhatsApp');
      }

      // Apply delay
      if (delayMs > 0) {
        await delay(delayMs);
      }

      // Since WhatsApp deprecated buttons for non-business accounts,
      // we send as a formatted text message with numbered options
      let formattedText = text + '\n\n';

      // Add buttons as numbered options
      buttons.forEach((btn, index) => {
        formattedText += `${index + 1}. ${btn.text}\n`;
      });

      // Add footer if provided
      if (footer) {
        formattedText += `\n_${footer}_`;
      }

      // Add instruction
      formattedText += '\n\n_Reply with the number of your choice_';

      const result = await client.socket.sendMessage(formattedReceiver, {
        text: formattedText,
      });

      logger.info(`Button message sent as formatted text`, {
        sessionId,
        receiver: formattedReceiver,
        messageId: result.key.id,
        buttonCount: buttons.length,
      });

      return {
        message: 'Button message sent successfully (as formatted text)',
        messageId: result.key.id,
        timestamp: result.messageTimestamp,
        note: 'Interactive buttons are deprecated by WhatsApp for non-business accounts. Message sent as numbered options.',
      };
    } catch (error) {
      logger.error(`Failed to send button message: ${error.message}`, { sessionId, receiver });
      throw error;
    }
  }

  /**
   * Send list/template message
   * Note: WhatsApp has deprecated interactive list messages for non-business accounts.
   * This method now sends lists as a formatted text message with sections and options.
   * For true interactive lists, use WhatsApp Business API (Cloud API).
   */
  async sendListMessage(sessionId, receiver, title, text, buttonText, sections, footer = '', delayMs = 0) {
    try {
      const client = SessionManager.get(sessionId);

      if (!client || !client.socket) {
        throw new Error('Session not found or not connected');
      }

      const formattedReceiver = formatPhone(receiver);

      // Check if number exists
      const check = await this.checkNumberExists(sessionId, receiver);
      if (!check.exists) {
        throw new Error('The receiver number does not exist on WhatsApp');
      }

      // Apply delay
      if (delayMs > 0) {
        await delay(delayMs);
      }

      // Since WhatsApp deprecated list messages for non-business accounts,
      // we send as a formatted text message with sections
      let formattedText = '';

      // Add title if provided
      if (title) {
        formattedText += `*${title}*\n\n`;
      }

      // Add main text
      formattedText += text + '\n\n';

      // Add sections with numbered options
      let optionNumber = 1;
      sections.forEach((section) => {
        if (section.title) {
          formattedText += `📋 *${section.title}*\n`;
        }
        if (section.rows && Array.isArray(section.rows)) {
          section.rows.forEach((row) => {
            formattedText += `${optionNumber}. ${row.title}`;
            if (row.description) {
              formattedText += ` - _${row.description}_`;
            }
            formattedText += '\n';
            optionNumber++;
          });
        }
        formattedText += '\n';
      });

      // Add footer if provided
      if (footer) {
        formattedText += `_${footer}_\n`;
      }

      // Add instruction
      formattedText += '\n_Reply with the number of your choice_';

      const result = await client.socket.sendMessage(formattedReceiver, {
        text: formattedText,
      });

      logger.info(`List message sent as formatted text`, {
        sessionId,
        receiver: formattedReceiver,
        messageId: result.key.id,
        sectionCount: sections.length,
      });

      return {
        message: 'List message sent successfully (as formatted text)',
        messageId: result.key.id,
        timestamp: result.messageTimestamp,
        note: 'Interactive lists are deprecated by WhatsApp for non-business accounts. Message sent as formatted options.',
      };
    } catch (error) {
      logger.error(`Failed to send list message: ${error.message}`, { sessionId, receiver });
      throw error;
    }
  }

  /**
   * Send bulk messages
   */
  async sendBulkMessages(sessionId, messages) {
    try {
      const client = SessionManager.get(sessionId);

      if (!client || !client.socket) {
        throw new Error('Session not found or not connected');
      }

      const results = [];
      const errors = [];

      for (let i = 0; i < messages.length; i++) {
        const msg = messages[i];

        try {
          const result = await this.sendTextMessage(
            sessionId,
            msg.receiver,
            msg.message.text,
            msg.delay || baileysConfig.message.bulkDelay
          );

          results.push({
            index: i,
            receiver: msg.receiver,
            success: true,
            messageId: result.messageId,
          });
        } catch (error) {
          errors.push({
            index: i,
            receiver: msg.receiver,
            error: error.message,
          });
        }
      }

      const sent = results.length;
      const failed = errors.length;
      const isAllFailed = failed === messages.length;

      return {
        message: isAllFailed
          ? 'Failed to send all messages'
          : failed > 0
            ? 'Some messages have been successfully sent'
            : 'All messages have been successfully sent',
        sent,
        failed,
        results,
        errors,
      };
    } catch (error) {
      logger.error(`Failed to send bulk messages: ${error.message}`, { sessionId });
      throw error;
    }
  }
}

export default new MessageService();
