import makeWASocket, {
  useMultiFileAuthState,
  DisconnectReason,
  Browsers,
  delay as baileysDelay,
  fetchLatestBaileysVersion,
} from '@whiskeysockets/baileys';
import { Boom } from '@hapi/boom';
import path from 'path';
import fs from 'fs';
import QRCode from 'qrcode';
import baileysConfig from '../config/baileys.config.js';
import logger from '../utils/logger.js';
import { retryWithBackoff } from '../utils/retry.js';

/**
 * WhatsApp Client - Baileys Wrapper
 */
class WhatsAppClient {
  constructor(sessionId) {
    this.sessionId = sessionId;
    this.socket = null;
    this.qrCode = null;
    this.retryCount = 0;
    this.isLegacy = false;
    this.authState = null;
    this.saveCreds = null;
  }

  /**
   * Initialize WhatsApp connection
   */
  async initialize(callbacks = {}) {
    try {
      const { onQR, onConnected, onDisconnected, onMessage } = callbacks;

      // Setup session directory
      const sessionPath = this.getSessionPath();
      if (!fs.existsSync(sessionPath)) {
        fs.mkdirSync(sessionPath, { recursive: true });
      }

      // Get latest Baileys version
      const { version } = await fetchLatestBaileysVersion();

      // Setup auth state
      const { state, saveCreds } = await useMultiFileAuthState(sessionPath);
      this.authState = state;
      this.saveCreds = saveCreds;

      // Create socket with proper browser fingerprint
      // Using Baileys built-in browser config to avoid "old version" warning
      this.socket = makeWASocket({
        version,
        auth: this.authState,
        printQRInTerminal: false,
        logger: logger.child({ class: 'Socket', sessionId: this.sessionId }),
        // Use Baileys official browser fingerprint - this is the most compatible option
        // Browsers.appropriate() returns the best browser config for current platform
        browser: Browsers.ubuntu('Chrome'),
        connectTimeoutMs: baileysConfig.connection.connectTimeout,
        keepAliveIntervalMs: baileysConfig.connection.keepAliveInterval,
        defaultQueryTimeoutMs: 60000,
        retryRequestDelayMs: 5000,
        markOnlineOnConnect: true,
        syncFullHistory: false,
        // Generate proper message IDs
        generateHighQualityLinkPreview: true,
      });

      // Event: Credentials update
      this.socket.ev.on('creds.update', saveCreds);

      // Event: Connection update
      this.socket.ev.on('connection.update', async (update) => {
        const { connection, lastDisconnect, qr } = update;

        // QR Code received
        if (qr) {
          try {
            this.qrCode = await QRCode.toDataURL(qr);
            logger.info(`QR Code generated for session: ${this.sessionId}`);

            if (onQR) {
              onQR(this.qrCode);
            }
          } catch (error) {
            logger.error(`Failed to generate QR code: ${error.message}`);
          }
        }

        // Connection opened
        if (connection === 'open') {
          this.retryCount = 0;
          this.qrCode = null;
          logger.info(`Connection opened for session: ${this.sessionId}`);

          if (onConnected) {
            const user = this.socket.user;
            onConnected(user);
          }
        }

        // Connection closed
        if (connection === 'close') {
          const shouldReconnect = this.handleDisconnect(lastDisconnect);

          logger.warn(`Connection closed for session: ${this.sessionId}`, {
            shouldReconnect,
            retryCount: this.retryCount,
          });

          if (onDisconnected) {
            onDisconnected(shouldReconnect);
          }

          if (shouldReconnect) {
            await this.reconnect(callbacks);
          }
        }
      });

      // Event: Messages
      if (onMessage) {
        this.socket.ev.on('messages.upsert', async ({ messages, type }) => {
          logger.info(`Messages upsert event received`, { sessionId: this.sessionId, type, count: messages.length });
          // Process both 'notify' and 'append' types - retry messages come as different types
          for (const msg of messages) {
            // Skip own messages and messages without content initially
            if (msg.key?.fromMe) {
              continue;
            }
            logger.info(`Processing incoming message`, {
              sessionId: this.sessionId,
              from: msg.key?.remoteJid,
              fromMe: msg.key?.fromMe,
              hasMessage: !!msg.message,
              type: type,
              messageStubType: msg.messageStubType
            });
            // Only forward messages with actual content (not decryption stubs)
            if (msg.message) {
              onMessage(msg);
            }
          }
        });

        // Also listen for message updates (retries with decrypted content)
        this.socket.ev.on('messages.update', async (updates) => {
          logger.info('Messages update event received', { sessionId: this.sessionId, count: updates.length });
          for (const update of updates) {
            // Check if this update includes decrypted content
            if (update.update?.message && !update.key?.fromMe) {
              logger.info('Message update with content received', {
                sessionId: this.sessionId,
                messageId: update.key?.id,
                hasMessage: !!update.update?.message
              });
              // Create a message object from the update
              const msg = {
                key: update.key,
                message: update.update.message,
                messageTimestamp: update.update.messageTimestamp || Math.floor(Date.now() / 1000),
                pushName: update.update.pushName
              };
              onMessage(msg);
            }
          }
        });
      }

      logger.info(`WhatsApp client initialized for session: ${this.sessionId}`);
    } catch (error) {
      logger.error(`Failed to initialize WhatsApp client: ${error.message}`, {
        sessionId: this.sessionId,
        error: error.stack,
      });
      throw error;
    }
  }

  /**
   * Handle disconnect reasons
   */
  handleDisconnect(lastDisconnect) {
    const statusCode = lastDisconnect?.error?.output?.statusCode;
    const reason = lastDisconnect?.error;

    logger.info(`Disconnect reason: ${statusCode}`, {
      sessionId: this.sessionId,
      error: reason?.message,
    });

    // Logged out - don't reconnect
    if (statusCode === DisconnectReason.loggedOut) {
      logger.warn(`Session logged out: ${this.sessionId}`);
      return false;
    }

    // Bad session - don't reconnect
    if (statusCode === DisconnectReason.badSession) {
      logger.warn(`Bad session: ${this.sessionId}`);
      return false;
    }

    // Max retries reached
    if (this.retryCount >= baileysConfig.connection.maxRetries) {
      logger.error(`Max retries reached for session: ${this.sessionId}`);
      return false;
    }

    // Reconnect for other reasons
    return true;
  }

  /**
   * Reconnect with exponential backoff
   */
  async reconnect(callbacks) {
    this.retryCount++;

    const delay = Math.min(
      baileysConfig.connection.reconnectInterval * Math.pow(2, this.retryCount - 1),
      30000
    );

    logger.info(`Reconnecting in ${delay}ms... (Attempt ${this.retryCount})`, {
      sessionId: this.sessionId,
    });

    await baileysDelay(delay);
    await this.initialize(callbacks);
  }

  /**
   * Get connection status
   */
  getStatus() {
    if (!this.socket) {
      return { status: 'disconnected', isSession: false };
    }

    const states = ['connecting', 'connected', 'disconnecting', 'disconnected'];
    let status = states[this.socket.ws?.readyState] || 'disconnected';

    // Check if authenticated via socket.user
    const hasUser = this.socket.user !== undefined;

    // Also check if we have valid credentials (creds.json with me field or registered)
    const hasValidCreds =
      (this.authState?.creds?.registered === true) ||
      (this.authState?.creds?.me !== undefined);

    // Consider authenticated if user is set OR we have valid credentials
    // Note: We check hasUser first because socket might be reconnecting (readyState temporarily undefined)
    const isAuthenticated = hasUser || (status === 'connected' && hasValidCreds);

    // Get user info from socket.user or from credentials
    let userInfo = this.socket.user || null;
    if (!userInfo && hasValidCreds && this.authState?.creds?.me) {
      userInfo = this.authState.creds.me;
    }

    return {
      status: isAuthenticated ? 'authenticated' : status,
      isSession: isAuthenticated,
      user: userInfo,
    };
  }

  /**
   * Logout and cleanup
   */
  async logout() {
    try {
      if (this.socket) {
        await this.socket.logout();
      }
    } catch (error) {
      logger.error(`Error during logout: ${error.message}`);
    } finally {
      this.cleanup();
    }
  }

  /**
   * Cleanup resources
   */
  cleanup() {
    try {
      // Close socket
      if (this.socket) {
        this.socket.end(undefined);
        this.socket = null;
      }

      // Delete session files
      const sessionPath = this.getSessionPath();
      if (fs.existsSync(sessionPath)) {
        fs.rmSync(sessionPath, { recursive: true, force: true });
      }

      logger.info(`Session cleaned up: ${this.sessionId}`);
    } catch (error) {
      logger.error(`Error during cleanup: ${error.message}`);
    }
  }

  /**
   * Get session directory path
   */
  getSessionPath() {
    return path.join(baileysConfig.storage.sessionsPath, `md_${this.sessionId}`);
  }
}

export default WhatsAppClient;
