import express from 'express';
import cors from 'cors';
import helmet from 'helmet';
import compression from 'compression';
import nodeCleanup from 'node-cleanup';
import serverConfig from './config/server.config.js';
import runtimeConfig from './config/runtime.config.js';
import logger from './utils/logger.js';
import routes from './routes/index.js';
import rateLimiter from './middleware/rateLimit.middleware.js';
import { errorHandler, notFoundHandler } from './middleware/errorHandler.middleware.js';
import { licenseMiddleware, initializeLicense } from './middleware/license.middleware.js';
import SessionManager from './core/SessionManager.js';

const app = express();

/**
 * Security Middleware
 */
app.use(helmet({
  contentSecurityPolicy: false,
}));

/**
 * Dynamic CORS Configuration
 * Uses runtime config that can be updated from Laravel
 */
app.use((req, res, next) => {
  const allowedOrigins = runtimeConfig.get('allowedOrigins');
  const origin = req.headers.origin;

  if (allowedOrigins.includes('*') || allowedOrigins.includes(origin)) {
    res.header('Access-Control-Allow-Origin', origin || '*');
    res.header('Access-Control-Allow-Credentials', 'true');
    res.header('Access-Control-Allow-Methods', 'GET,HEAD,PUT,PATCH,POST,DELETE,OPTIONS');
    res.header('Access-Control-Allow-Headers', 'Origin, X-Requested-With, Content-Type, Accept, Authorization, X-API-Key');
  }

  if (req.method === 'OPTIONS') {
    return res.sendStatus(200);
  }

  next();
});

/**
 * Body Parsing Middleware
 */
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true, limit: '10mb' }));

/**
 * Compression Middleware
 */
app.use(compression());

/**
 * Rate Limiting (applies to all routes)
 */
app.use(rateLimiter);

/**
 * License Verification Middleware
 * Validates license for protected routes
 */
app.use(licenseMiddleware);

/**
 * Request Logging
 */
app.use((req, res, next) => {
  logger.info(`${req.method} ${req.path}`, {
    ip: req.ip,
    userAgent: req.get('user-agent'),
  });
  next();
});

/**
 * API Routes
 */
app.use('/', routes);

/**
 * 404 Handler
 */
app.use(notFoundHandler);

/**
 * Global Error Handler
 */
app.use(errorHandler);

/**
 * Start Server
 */
const server = app.listen(serverConfig.port, serverConfig.host, async () => {
  logger.info(`🚀 XSender WhatsApp Service started`);

  // Initialize license verification
  await initializeLicense();

  // Restore existing sessions
  logger.info('Restoring existing sessions...');
  await SessionManager.restoreSessions();
});

/**
 * Graceful Shutdown Handler
 */
const gracefulShutdown = async (signal) => {
  logger.info(`${signal} received. Starting graceful shutdown...`);

  // Close server
  server.close(async () => {
    logger.info('HTTP server closed');

    // Cleanup sessions
    await SessionManager.cleanup();

    logger.info('Graceful shutdown completed');
    process.exit(0);
  });

  // Force shutdown after 30 seconds
  setTimeout(() => {
    logger.error('Forceful shutdown after timeout');
    process.exit(1);
  }, 30000);
};

/**
 * Process Event Handlers
 */
process.on('SIGTERM', () => gracefulShutdown('SIGTERM'));
process.on('SIGINT', () => gracefulShutdown('SIGINT'));

process.on('uncaughtException', (error) => {
  logger.error('Uncaught Exception:', {
    error: error.message,
    stack: error.stack,
  });
  gracefulShutdown('uncaughtException');
});

process.on('unhandledRejection', (reason, promise) => {
  logger.error('Unhandled Rejection:', {
    reason,
    promise,
  });
});

/**
 * Node Cleanup Hook
 */
nodeCleanup(async (exitCode, signal) => {
  logger.info('Cleanup before exit', { exitCode, signal });
  await SessionManager.cleanup();
});

export default app;
